document.addEventListener('DOMContentLoaded', () => {
    // New Element IDs based on the "Kolichala" style template
    const unicodeInput = document.getElementById('unicodeInput');
    const anuOutput = document.getElementById('anuOutput');
    const convertBtn = document.getElementById('convertBtn'); // Manual trigger
    const mappingSelect = document.getElementById('mappingSelect'); // Dropdown
    const copyBtn = document.getElementById('copyBtn');
    const copyMsg = document.getElementById('copyMsg');

    // Debugging: Check if mappings are loaded
    if (!window.anu7Mapping) {
        console.error('CRITICAL: window.anu7Mapping is Undefined! Check if anu7.js is loaded.');
        alert('Error: Anu 7 Mapping logic not loaded. Please Hard Refresh (Ctrl+F5).');
    }
    if (!window.anu6Mapping) {
        console.warn('WARNING: window.anu6Mapping is Undefined! Anu 6 selection will fail.');
    }

    // Default to Anu 7
    let currentMapping = window.anu7Mapping || {};

    // -----------------------------------------------------
    // MAPPING SWITCHER
    // -----------------------------------------------------
    mappingSelect.addEventListener('change', (e) => {
        const selectedValue = e.target.value;
        console.log('Dropdown changed to:', selectedValue);

        if (selectedValue === 'anu6') {
            if (!window.anu6Mapping) {
                alert('Anu 6 Mapping not found! Check console.');
                return;
            }
            currentMapping = window.anu6Mapping;
            document.getElementById('outputLabel').textContent = 'Anu 6 Output';
            console.log('Switched to Anu 6 Mapping');
        } else {
            currentMapping = window.anu7Mapping;
            document.getElementById('outputLabel').textContent = 'Anu 7 Output';
            console.log('Switched to Anu 7 Mapping');
        }

        // Retrigger conversion specifically if there is text
        if (unicodeInput.value) {
            triggerConversion();
        }
    });

    // -----------------------------------------------------
    // CORE CONVERSION LOGIC
    // -----------------------------------------------------

    function convertToAnu(uniText) {
        if (!uniText) return '';

        // Destructure current mapping
        const { vowels, extensions, consonants } = currentMapping;

        let result = "";

        for (let i = 0; i < uniText.length; i++) {
            let code = uniText.charCodeAt(i);

            // Pass through strict ASCII (English/Numbers)
            if (code < 3073 || code > 3183) {
                result += String.fromCharCode(code);
                continue;
            }

            // Space
            if (code === 0x20) {
                result += " ";
                continue;
            }
            // Enter
            if (code === 0xA) {
                result += "\n";
                continue;
            }

            // Vowels
            if (code >= 3074 && code <= 3092) {
                if (vowels[code]) {
                    result += vowels[code];
                    continue;
                }
            }

            // Consonants
            if (code >= 3093 && code <= 3129) {
                let charData = consonants[code];
                if (!charData) {
                    continue;
                }

                let totalLetter = "";

                // Check for Virama (Halant) -> Potential Conjunct (Vattu)
                if (uniText.charCodeAt(i + 1) == 3149) {
                    let extensionNumber = uniText.charCodeAt(i + 2);

                    // Case 1: Consonant + Virama + Consonant + Matra (Combos)
                    if (extensionNumber && uniText.charCodeAt(i + 3) >= 3134 && uniText.charCodeAt(i + 3) <= 3150) {
                        let symbolCode = uniText.charCodeAt(i + 3);
                        if (charData.symbols[symbolCode]) {
                            // SPECIAL RA HANDLING (Kra etc) - Logic might be same for both, but char codes differ in mapping
                            if (extensionNumber === 3120) {
                                totalLetter = (extensions[extensionNumber] || "") + charData.symbols[symbolCode];
                            } else {
                                totalLetter = charData.symbols[symbolCode] + (extensions[extensionNumber] || "");
                            }
                            i += 3;
                        }
                    }
                    // Case 2: Consonant + Virama + Consonant (Simple Vattu)
                    else if (!uniText.charCodeAt(i + 2) || uniText.charCodeAt(i + 2) < 3093 || uniText.charCodeAt(i + 2) > 3129) {
                        totalLetter = charData.symbols[3149];
                        // Special Ra Case
                        if (extensionNumber === 3120) {
                            totalLetter = (extensions[extensionNumber] || "") + charData.base;
                        }
                    }
                    else {
                        // Case 3: Consonant + Virama + Consonant (Base + Vattu)
                        if (extensionNumber === 3120) {
                            totalLetter = (extensions[extensionNumber] || "") + charData.base;
                        } else {
                            totalLetter = charData.base + (extensions[extensionNumber] || "");
                        }
                        i += 2;
                    }

                } else {
                    // No Virama. Check if next is a Symbol (Matra)
                    let nextCode = uniText.charCodeAt(i + 1);
                    if (nextCode && nextCode >= 3134 && nextCode <= 3150) {
                        if (charData.symbols[nextCode]) {
                            totalLetter = charData.symbols[nextCode];
                            i += 1;
                        }
                    } else {
                        // Just the base consonant
                        totalLetter = charData.base;
                    }
                }

                if (totalLetter) {
                    result += totalLetter;
                }
            }
        }
        return result;
    }

    // -----------------------------------------------------
    // TRIGGER CONVERSION (Shared)
    // -----------------------------------------------------
    function triggerConversion() {
        const inputText = unicodeInput.value;
        const convertedText = convertToAnu(inputText);
        anuOutput.value = convertedText;
    }

    // -----------------------------------------------------
    // EVENT LISTENERS
    // -----------------------------------------------------

    // Real-time conversion
    unicodeInput.addEventListener('input', triggerConversion);

    // Manual Convert Button
    convertBtn.addEventListener('click', triggerConversion);

    // Copy function
    copyBtn.addEventListener('click', () => {
        if (!anuOutput.value) return;

        anuOutput.select();
        anuOutput.setSelectionRange(0, 99999);

        navigator.clipboard.writeText(anuOutput.value).then(() => {
            copyMsg.style.display = 'inline-block';
            setTimeout(() => {
                copyMsg.style.display = 'none';
            }, 2000);
        }).catch(err => {
            console.error('Failed to copy: ', err);
        });
    });
});

